# Copyright 2020 by John A Kline <john@johnkline.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

"""
WeeWX module that records AirLink air quality sensor readings.
"""

import logging
import math
import requests
import sys
import threading
import time

from dataclasses import dataclass
from typing import Any, Dict, List, Tuple

import weeutil.weeutil
import weewx
import weewx.units
import weewx.xtypes

from weewx.units import ValueTuple
from weeutil.weeutil import timestamp_to_string
from weeutil.weeutil import to_bool
from weeutil.weeutil import to_int
from weewx.engine import StdService

log = logging.getLogger(__name__)

WEEWX_AIRLINK_VERSION = "1.31"

if sys.version_info[0] < 3 or (sys.version_info[0] == 3 and sys.version_info[1] < 7):
    raise weewx.UnsupportedFeature(
        "weewx-airlink requires Python 3.7 or later, found %s.%s" % (sys.version_info[0], sys.version_info[1]))

if weewx.__version__ < "4":
    raise weewx.UnsupportedFeature(
        "weewx-airlink requires WeeWX 4, found %s" % weewx.__version__)

# Set up observation types not in weewx.units

weewx.units.USUnits['air_quality_index']       = 'aqi'
weewx.units.MetricUnits['air_quality_index']   = 'aqi'
weewx.units.MetricWXUnits['air_quality_index'] = 'aqi'

weewx.units.USUnits['air_quality_color']       = 'aqi_color'
weewx.units.MetricUnits['air_quality_color']   = 'aqi_color'
weewx.units.MetricWXUnits['air_quality_color'] = 'aqi_color'

weewx.units.default_unit_label_dict['aqi']  = ' AQI'
weewx.units.default_unit_label_dict['aqi_color'] = ' RGB'

weewx.units.default_unit_format_dict['aqi']  = '%d'
weewx.units.default_unit_format_dict['aqi_color'] = '%d'

weewx.units.obs_group_dict['pm1_0_1m'] = 'group_concentration'
weewx.units.obs_group_dict['pm10_0_1m'] = 'group_concentration'
weewx.units.obs_group_dict['pm2_5_aqi'] = 'air_quality_index'
weewx.units.obs_group_dict['pm2_5_aqi_color'] = 'air_quality_color'
weewx.units.obs_group_dict['pm2_5_1m'] = 'group_concentration'
weewx.units.obs_group_dict['pm2_5_1m_aqi'] = 'air_quality_index'
weewx.units.obs_group_dict['pm2_5_1m_aqi_color'] = 'air_quality_color'

weewx.units.obs_group_dict['temp'] = 'group_temperature'
weewx.units.obs_group_dict['co2_Temp'] = 'group_temperature'
weewx.units.obs_group_dict['pm10_0_nowcast'] = 'group_concentration'
weewx.units.obs_group_dict['dew_point'] = 'group_temperature'
weewx.units.obs_group_dict['wet_bulb'] = 'group_temperature'
weewx.units.obs_group_dict['heat_index'] = 'group_temperature'
weewx.units.obs_group_dict['hum'] = 'group_percent'
weewx.units.obs_group_dict['co2_Hum'] = 'group_percent'
weewx.units.obs_group_dict['dewpoint1'] = 'group_temperature'
weewx.units.obs_group_dict['wetbulb1'] = 'group_temperature'
weewx.units.obs_group_dict['heatindex1'] = 'group_temperature'

weewx.units.obs_group_dict['pm10_0_nowcast'] = 'group_concentration'
weewx.units.obs_group_dict['pm2_5_nowcast'] = 'group_concentration'
weewx.units.obs_group_dict['pm_2p5_last_1_hour'] = 'group_concentration'
weewx.units.obs_group_dict['pm_2p5_last_3_hours'] = 'group_concentration'
weewx.units.obs_group_dict['pm_2p5_last_24_hours'] = 'group_concentration'
weewx.units.obs_group_dict['pm_10_last_1_hour'] = 'group_concentration'
weewx.units.obs_group_dict['pm_10_last_3_hours'] = 'group_concentration'
weewx.units.obs_group_dict['pm_10_last_24_hours'] = 'group_concentration'

weewx.units.obs_group_dict['pct_pm_data_last_1_hour'] = 'group_percent'
weewx.units.obs_group_dict['pct_pm_data_last_3_hours'] = 'group_percent'
weewx.units.obs_group_dict['pct_pm_data_nowcast'] = 'group_percent'
weewx.units.obs_group_dict['pct_pm_data_last_24_hours'] = 'group_percent'


class Source:
    def __init__(self, config_dict, name):
        # Raise KeyEror if name not in dictionary.
        source_dict = config_dict[name]
        self.enable = to_bool(source_dict.get('enable', False))
        self.hostname = source_dict.get('hostname', '')
        self.port = to_int(source_dict.get('port', 80))
        self.timeout  = to_int(source_dict.get('timeout', 10))

@dataclass
class Concentrations:
    timestamp     : float
    pm_1_last     : float
    pm_2p5_last   : float
    pm_10_last    : float
    pm_1          : float
    pm_2p5        : float
    pm_10         : float
    pm_2p5_nowcast: float
    pm_10_nowcast : float
    hum           : float
    temp          : float
    dew_point     : float
    wet_bulb      : float
    heat_index    : float
    co2_Temp      : float
    co2_Hum       : float
    dewpoint1     : float
    wetbulb1      : float
    heatindex1    : float
    pm_2p5_last_1_hour : float
    pm_2p5_last_3_hours : float
    pm_2p5_last_24_hours : float
    pm_10_last_1_hour : float
    pm_10_last_3_hours : float
    pm_10_last_24_hours : float
    pct_pm_data_last_1_hour   : float 
    pct_pm_data_last_3_hours  : float
    pct_pm_data_last_24_hours : float

@dataclass
class Configuration:
    lock            : threading.Lock
    concentrations  : Concentrations # Controlled by lock
    archive_interval: int            # Immutable
    archive_delay   : int            # Immutable
    poll_interval   : int            # Immutable
    sources         : List[Source]   # Immutable

def get_concentrations(cfg: Configuration):
    for source in cfg.sources:
        if source.enable:
            record = collect_data(source.hostname,
                                  source.port,
                                  source.timeout,
                                  cfg.archive_interval)
            if record is not None:
                log.debug('get_concentrations: source: %s' % record)
                reading_ts = to_int(record['dateTime'])
                age_of_reading = time.time() - reading_ts
                if age_of_reading > cfg.archive_interval:
                    log.info('Reading from %s:%d is old: %d seconds.' % (
                        source.hostname, source.port, age_of_reading))
                    continue
                log.debug('get_concentrations: record: %s' % record)
                concentrations = Concentrations(
                    timestamp      = reading_ts,
                    pm_1_last      = record['pm_1_last'],
                    pm_2p5_last    = record['pm_2p5_last'],
                    pm_10_last     = record['pm_10_last'],
                    pm_1           = record['pm_1'],
                    pm_2p5         = record['pm_2p5'],
                    pm_10          = record['pm_10'],
                    pm_2p5_nowcast = record['pm_2p5_nowcast'],
                    pm_10_nowcast  = record['pm_10_nowcast'],
                    hum            = record['hum'],
                    temp           = record['temp'],
                    co2_Hum        = record['co2_Hum'],
                    co2_Temp       = record['co2_Temp'],
                    dew_point      = record['dew_point'],
                    wet_bulb       = record['wet_bulb'],
                    heat_index     = record['heat_index'],
                    dewpoint1      = record['dewpoint1'],
                    wetbulb1       = record['wetbulb1'],
                    heatindex1     = record['heatindex1'],
                    pm_2p5_last_1_hour = record['pm_2p5_last_1_hour'],
                    pm_2p5_last_3_hours = record['pm_2p5_last_3_hours'],
                    pm_2p5_last_24_hours = record['pm_2p5_last_24_hours'],
                    pm_10_last_1_hour = record['pm_10_last_1_hour'],
                    pm_10_last_3_hours = record['pm_10_last_3_hours'],
                    pm_10_last_24_hours = record['pm_10_last_24_hours'],
                    pct_pm_data_last_1_hour = record['pct_pm_data_last_1_hour'], 
                    pct_pm_data_last_3_hours = record['pct_pm_data_last_3_hours'],
                    pct_pm_data_last_24_hours = record['pct_pm_data_last_24_hours'],
                )
                log.debug('get_concentrations: concentrations: %s' % concentrations)
                return concentrations
    log.error('Could not get concentrations from any source.')
    return None

def is_type(j: Dict[str, Any], t, name: str, none_ok: bool = False) -> bool:
    try:
        x = j[name]
        if x is None and none_ok:
            return True
        if not isinstance(x, t):
            log.debug('%s is not an instance of %s: %s' % (name, t, j[name]))
            return False
        return True
    except KeyError as e:
        log.debug('is_type: could not find key: %s' % e)
        return False
    except Exception as e:
        log.debug('is_type: exception: %s' % e)
        return False

def convert_data_structure_type_5_to_6(j: Dict[str, Any]) -> None:
    # Fix up these names and change data_structure_type to 6
    try:
        j['data']['conditions'][0]['pm_10'] = j['data']['conditions'][0]['pm_10p0']
        j['data']['conditions'][0]['pm_10p0'] = None
        j['data']['conditions'][0]['pm_10_last_1_hour'] = j['data']['conditions'][0]['pm_10p0_last_1_hour']
        j['data']['conditions'][0]['pm_10p0_last_1_hour'] = None
        j['data']['conditions'][0]['pm_10_last_3_hours'] = j['data']['conditions'][0]['pm_10p0_last_3_hours']
        j['data']['conditions'][0]['pm_10p0_last_3_hours'] = None
        j['data']['conditions'][0]['pm_10_last_24_hours'] = j['data']['conditions'][0]['pm_10p0_last_24_hours']
        j['data']['conditions'][0]['pm_10p0_last_24_hours'] = None
        j['data']['conditions'][0]['pm_10_nowcast'] = j['data']['conditions'][0]['pm_10p0_nowcast']
        j['data']['conditions'][0]['pm_10p0_nowcast'] = None

        j['data']['conditions'][0]['data_structure_type'] = 6
        log.debug('Converted type 5 record to type 6.')
    except Exception as e:
        log.info('convert_data_structure_type_5_to_6: exception: %s' % e)
        # Let sanity check handle the issue.

def is_sane(j: Dict[str, Any]) -> Tuple[bool, str]:
    if j['error'] is not None:
        return False, 'Error: %s' % j['error']

    if not is_type(j, dict, 'data'):
        return False, 'Missing or malformed "data" field'

    if not is_type(j['data'], str, 'name'):
        return False, 'Missing or malformed "name" field'

    if not is_type(j['data'], int, 'ts'):
        return False, 'Missing or malformed "ts" field'

    if not is_type(j['data'], list, 'conditions'):
        return False, 'Missing or malformed "conditions" field'

    if len(j['data']['conditions']) == 0:
        return False, 'Expected one element in conditions array.'

    if not is_type(j['data']['conditions'][0], int, 'data_structure_type'):
        return False, 'Missing or malformed "data_structure_type" field'

    if j['data']['conditions'][0]['data_structure_type'] != 6:
        return False, 'Expected data_structure_type of 6 (or type 5 auto converted to 6.'

    for name in ['pm_1_last', 'pm_2p5_last', 'pm_10_last', 'last_report_time',
            'pct_pm_data_last_1_hour', 'pct_pm_data_last_3_hours',
            'pct_pm_data_nowcast', 'pct_pm_data_last_24_hours']:
        if not is_type(j['data']['conditions'][0], int, name, True):
            return False, 'Missing or malformed "%s" field' % name

    if not is_type(j['data']['conditions'][0], int, 'lsid', True):
        return False, 'Missing or malformed "lsid" field'

    for name in ['temp', 'hum', 'dew_point', 'wet_bulb', 'heat_index']:
        if not is_type(j['data']['conditions'][0], float, name):
            return False, 'Missing or malformed "%s" field' % name

    for name in ['pm_1', 'pm_2p5', 'pm_2p5_last_1_hour',
             'pm_2p5_last_3_hours', 'pm_2p5_last_24_hours', 'pm_2p5_nowcast',
             'pm_10', 'pm_10_last_1_hour', 'pm_10_last_3_hours',
             'pm_10_last_24_hours', 'pm_10_nowcast']:
        if not is_type(j['data']['conditions'][0], float, name, True):
            return False, 'Missing or malformed "%s" field' % name

    return True, ''

def collect_data(hostname, port, timeout, archive_interval):

    j = None
    url = 'http://%s:%s/v1/current_conditions' % (hostname, port)

    try:
        # fetch data
        log.debug('collect_data: fetching from url: %s, timeout: %d' % (url, timeout))
        r = requests.get(url=url, timeout=timeout)
        r.raise_for_status()
        log.debug('collect_data: %s returned %r' % (hostname, r))
        if r:
            # convert to json
            j = r.json()
            log.debug('collect_data: json returned from %s is: %r' % (hostname, j))
            # Check for error
            if 'error' in j and j['error'] is not None:
                error = j['error']
                code = error['code']
                message = error['message']
                log.info('%s returned error(%d): %s' % (url, code, message))
                return None
            # If data structure type 5, convert it to 6.
            if j['data']['conditions'][0]['data_structure_type'] == 5:
                convert_data_structure_type_5_to_6(j)
            # Check for sanity
            sane, msg = is_sane(j)
            if not sane:
                log.info('Reading not sane:  %s (%s)' % (msg, j))
                return None
            time_of_reading = j['data']['conditions'][0]['last_report_time']
            # The reading could be old.
            # Check that it's not older than now - arcint
            age_of_reading = time.time() - time_of_reading
            if age_of_reading > archive_interval:
                # Perhaps the AirLink has rebooted.  If so, the last_report_time
                # will be seconds from boot time (until the device syncs
                # time.  Check for this by checking if concentrations.pm_1
                # is None.
                if j['data']['conditions'][0]['pm_1'] is None:
                    log.info('last_report_time must be time since boot: %d seconds.  Record: %s'
                             % (time_of_reading, j))
                else:
                    # Not current.  (Note: Rarely, spurious timestamps (e.g., 2016 in 2020)
                    # have been observed.  Both the ts and last_report_time fields are incorrect.
                    # Example on Oct 10 21:11:38:
                    # {'data': {'did': '001D0A100214', 'name': 'airlink', 'ts': 1461926887,
                    # 'conditions': [{'lsid': 349506, 'data_structure_type': 6, 'temp': 67.7,
                    # 'hum': 72.2, 'dew_point': 58.4, 'wet_bulb': 61.2, 'heat_index': 68.1,
                    # 'pm_1_last': 0, 'pm_2p5_last': 0, 'pm_10_last': 0, 'pm_1': 0.0,
                    # 'pm_2p5': 0.0, 'pm_2p5_last_1_hour': 0.13, 'pm_2p5_last_3_hours': 0.27,
                    # 'pm_2p5_last_24_hours': 0.43, 'pm_2p5_nowcast': 0.23, 'pm_10': 1.09,
                    # 'pm_10_last_1_hour': 0.64, 'pm_10_last_3_hours': 0.89,
                    # 'pm_10_last_24_hours': 1.02, 'pm_10_nowcast': 0.84,
                    # 'last_report_time': 1461926886, 'pct_pm_data_last_1_hour': 100,
                    # 'pct_pm_data_last_3_hours': 100, 'pct_pm_data_nowcast': 100,
                    # 'pct_pm_data_last_24_hours': 100}]}, 'error': None}
                    log.info('Ignoring reading from %s--age: %d seconds.  Record: %s'
                             % (hostname, age_of_reading, j))
                j = None
    except Exception as e:
        log.info('collect_data: Attempt to fetch from: %s failed: %s.' % (hostname, e))
        j = None


    if j is None:
        return None

    # create a record
    log.debug('Successful read from %s.' % hostname)
    return populate_record(time_of_reading, j)

def populate_record(ts, j):
    record = dict()
    record['dateTime'] = ts
    record['usUnits'] = weewx.US

    # put items into record
    missed = []

    def get_and_update_missed(key):
        if key in j['data']['conditions'][0]:
            return j['data']['conditions'][0][key]
        else:
            missed.append(key)
            return None

    record['last_report_time'] = get_and_update_missed('last_report_time')
    record['temp'] = get_and_update_missed('temp')
    record['hum'] = get_and_update_missed('hum')
    record['co2_Temp'] = record['temp']
    record['co2_Hum'] = record['hum']
    record['dew_point'] = get_and_update_missed('dew_point')
    record['wet_bulb'] = get_and_update_missed('wet_bulb')
    record['heat_index'] = get_and_update_missed('heat_index')
    record['dewpoint1'] = record['dew_point']
    record['wetbulb1'] = record['wet_bulb']
    record['heatindex1'] = record['heat_index']
    record['pct_pm_data_last_1_hour'] = get_and_update_missed('pct_pm_data_last_1_hour')
    record['pct_pm_data_last_3_hours'] = get_and_update_missed('pct_pm_data_last_3_hours')
    record['pct_pm_data_nowcast'] = get_and_update_missed('pct_pm_data_nowcast')
    record['pct_pm_data_last_24_hours'] = get_and_update_missed('pct_pm_data_last_24_hours')

    record['pm1_0'] = get_and_update_missed('pm_1_last')
    record['pm2_5'] = get_and_update_missed('pm_2p5_last')
    record['pm10_0'] = get_and_update_missed('pm_10_last')

    # Copy in all of the concentrations.
    record['pm_1'] = get_and_update_missed('pm_1')
    record['pm_1_last'] = get_and_update_missed('pm_1_last')
    for prefix in ['pm_2p5', 'pm_10']:
        key = prefix + '_last'
        record[key] = get_and_update_missed(key)
        key = prefix
        record[key] = get_and_update_missed(key)
        key = prefix + '_last_1_hour'
        record[key] = get_and_update_missed(key)
        key = prefix + '_last_3_hours'
        record[key] = get_and_update_missed(key)
        key = prefix + '_last_24_hours'
        record[key] = get_and_update_missed(key)
        key = prefix + '_nowcast'
        record[key] = get_and_update_missed(key)

    if missed:
        log.info("Sensor didn't report field(s): %s" % ','.join(missed))

    return record

class AirLink(StdService):
    """Collect AirLink air quality measurements."""

    def __init__(self, engine, config_dict):
        super(AirLink, self).__init__(engine, config_dict)
        log.info("Service version is %s." % WEEWX_AIRLINK_VERSION)

        self.engine = engine
        self.config_dict = config_dict.get('AirLink', {})

        self.cfg = Configuration(
            lock             = threading.Lock(),
            concentrations   = None,
            archive_interval = int(config_dict['StdArchive']['archive_interval']),
            archive_delay    = to_int(config_dict['StdArchive'].get('archive_delay', 15)),
            poll_interval    = 5,
            sources          = AirLink.configure_sources(self.config_dict))
        with self.cfg.lock:
            self.cfg.concentrations = get_concentrations(self.cfg)

        source_count = 0
        for source in self.cfg.sources:
            if source.enable:
                source_count += 1
                log.info(
                    'Source %d for AirLink readings: %s:%s, timeout: %d' % (
                    source_count, source.hostname, source.port, source.timeout))
        if source_count == 0:
            log.error('No sources configured for airlink extension.  AirLink extension is inoperable.')
        else:
            weewx.xtypes.xtypes.append(AQI())

            # Start a thread to query devices.
            dp: DevicePoller = DevicePoller(self.cfg)
            t: threading.Thread = threading.Thread(target=dp.poll_device)
            t.setName('AirLink')
            t.setDaemon(True)
            t.start()

            self.bind(weewx.NEW_LOOP_PACKET, self.new_loop_packet)

    def new_loop_packet(self, event):
        log.debug('new_loop_packet(%s)' % event)
        AirLink.fill_in_packet(self.cfg, event.packet)

    @staticmethod
    def fill_in_packet(cfg: Configuration, packet: Dict):
        with cfg.lock:
            log.debug('new_loop_packet: cfg.concentrations: %s' % cfg.concentrations)
            if cfg.concentrations is not None and \
                    cfg.concentrations.timestamp is not None and \
                    cfg.concentrations.timestamp + \
                    cfg.archive_interval >= time.time():
                log.debug('Time of reading being inserted: %s' % timestamp_to_string(cfg.concentrations.timestamp))
                if cfg.concentrations.hum is not None:
                   packet['hum']       = cfg.concentrations.hum
                   packet['co2_Hum']   = packet['hum']
                if cfg.concentrations.temp is not None:
                   packet['temp']       = cfg.concentrations.temp
                   packet['co2_Temp']   = packet['temp']
                if cfg.concentrations.heat_index is not None:
                   packet['heat_index']   = cfg.concentrations.heat_index
                   packet['heatindex1']   = packet['heat_index']
                if cfg.concentrations.dew_point is not None:
                   packet['dew_point']   = cfg.concentrations.dew_point
                   packet['dewpoint1']   = packet['dew_point']
                if cfg.concentrations.wet_bulb is not None:
                   packet['wet_bulb']   = cfg.concentrations.wet_bulb
                   packet['wetbulb1']   = packet['wet_bulb']

                if cfg.concentrations.pm_2p5_last_24_hours is not None:
                   packet['pm_2p5_last_24_hours']   = cfg.concentrations.pm_2p5_last_24_hours
                if cfg.concentrations.pm_2p5_last_3_hours is not None:
                   packet['pm_2p5_last_3_hours']   = cfg.concentrations.pm_2p5_last_3_hours
                if cfg.concentrations.pm_2p5_last_1_hour is not None:
                   packet['pm_2p5_last_1_hour']   = cfg.concentrations.pm_2p5_last_1_hour

                if cfg.concentrations.pm_10_last_24_hours is not None:
                   packet['pm_10_last_24_hours']   = cfg.concentrations.pm_10_last_24_hours
                if cfg.concentrations.pm_10_last_3_hours is not None:
                   packet['pm_10_last_3_hours']   = cfg.concentrations.pm_10_last_3_hours
                if cfg.concentrations.pm_10_last_1_hour is not None:
                   packet['pm_10_last_1_hour']   = cfg.concentrations.pm_10_last_1_hour

                if cfg.concentrations.pct_pm_data_last_24_hours is not None:
                   packet['pct_pm_data_last_24_hours']   = cfg.concentrations.pct_pm_data_last_24_hours
                if cfg.concentrations.pct_pm_data_last_3_hours is not None:
                   packet['pct_pm_data_last_3_hours']   = cfg.concentrations.pct_pm_data_last_3_hours
                if cfg.concentrations.pct_pm_data_last_1_hour is not None:
                   packet['pct_pm_data_last_1_hour']   = cfg.concentrations.pct_pm_data_last_1_hour
 
                if cfg.concentrations.pm_1_last is not None:
                # Insert pm1_0, pm2_5, pm10_0, aqi and aqic into loop packet.
                    packet['pm1_0'] = cfg.concentrations.pm_1_last
                    log.debug('Inserted packet[pm1_0]: %f into packet.' % cfg.concentrations.pm_1_last)
                if cfg.concentrations.pm_2p5_last is not None:
                    packet['pm2_5'] = cfg.concentrations.pm_2p5_last
                    log.debug('Inserted packet[pm2_5]: %f into packet.' % cfg.concentrations.pm_2p5_last)
                    # Put aqi and color in the packet.
                    packet['pm2_5_aqi'] = AQI.compute_pm2_5_aqi(packet['pm2_5'])
                    packet['pm2_5_aqi_color'] = AQI.compute_pm2_5_aqi_color(packet['pm2_5_aqi'])
                if cfg.concentrations.pm_10_last is not None:
                    packet['pm10_0'] = cfg.concentrations.pm_10_last
                    log.debug('Inserted packet[pm10_0]: %f into packet.' % cfg.concentrations.pm_10_last)

                # Also insert one minute averages as these averages are more useful for showing in realtime.
                # If 1m averages are not available, use last instead.
                if cfg.concentrations.pm_1 is not None:
                    packet['pm1_0_1m']       = cfg.concentrations.pm_1
                elif cfg.concentrations.pm_1_last is not None:
                    packet['pm1_0_1m']       = cfg.concentrations.pm_1_last
                if cfg.concentrations.pm_2p5 is not None:
                    packet['pm2_5_1m']       = cfg.concentrations.pm_2p5
                elif cfg.concentrations.pm_2p5_last is not None:
                    packet['pm2_5_1m']       = cfg.concentrations.pm_2p5_last
                if cfg.concentrations.pm_10 is not None:
                    packet['pm10_0_1m']      = cfg.concentrations.pm_10
                elif cfg.concentrations.pm_10_last is not None:
                    packet['pm10_0_1m']      = cfg.concentrations.pm_10_last

                # Add 1m aqi and color
                if 'pm2_5_1m' in packet:
                    packet['pm2_5_1m_aqi'] = AQI.compute_pm2_5_aqi(packet['pm2_5_1m'])
                    packet['pm2_5_1m_aqi_color'] = AQI.compute_pm2_5_aqi_color(packet['pm2_5_1m_aqi'])

                # And insert nowcast for pm 2.5 and 10 as some might want to report that.
                # If nowcast not available, don't substitute.
                if cfg.concentrations.pm_2p5_nowcast is not None:
                    packet['pm2_5_nowcast']  = cfg.concentrations.pm_2p5_nowcast
                    packet['pm2_5_nowcast_aqi'] = AQI.compute_pm2_5_aqi(packet['pm2_5_nowcast'])
                    packet['pm2_5_nowcast_aqi_color'] = AQI.compute_pm2_5_aqi_color(packet['pm2_5_nowcast_aqi'])
                if cfg.concentrations.pm_10_nowcast is not None:
                    packet['pm10_0_nowcast'] = cfg.concentrations.pm_10_nowcast
            else:
                log.error('Found no concentrations to insert.')

    def configure_sources(config_dict):
        sources = []
        idx = 0
        while True:
            idx += 1
            try:
                source = Source(config_dict, 'Sensor%d' % idx)
                sources.append(source)
            except KeyError:
                break

        return sources

class DevicePoller:
    def __init__(self, cfg: Configuration):
        self.cfg = cfg

    def poll_device(self) -> None:
        log.debug('poll_device: start')
        while True:
            try:
                log.debug('poll_device: calling get_concentrations.')
                concentrations = get_concentrations(self.cfg)
            except Exception as e:
                log.error('poll_device exception: %s' % e)
                weeutil.logger.log_traceback(log.critical, "    ****  ")
                concentrations = None
            log.debug('poll_device: concentrations: %s' % concentrations)
            if concentrations is not None:
                with self.cfg.lock:
                    self.cfg.concentrations = concentrations
            log.debug('poll_device: Sleeping for %d seconds.' % self.cfg.poll_interval)
            time.sleep(self.cfg.poll_interval)

class AQI(weewx.xtypes.XType):
    """
    AQI XType which computes the AQI (air quality index) from
    the pm2_5 value.
    """

    def __init__(self):
        pass

    agg_sql_dict = {
        'avg': "SELECT AVG(pm2_5), usUnits FROM %(table_name)s "
               "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL",
        'count': "SELECT COUNT(dateTime), usUnits FROM %(table_name)s "
                 "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL",
        'first': "SELECT pm2_5, usUnits FROM %(table_name)s "
                 "WHERE dateTime = (SELECT MIN(dateTime) FROM %(table_name)s "
                 "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL",
        'last': "SELECT pm2_5, usUnits FROM %(table_name)s "
                "WHERE dateTime = (SELECT MAX(dateTime) FROM %(table_name)s "
                "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL",
        'min': "SELECT pm2_5, usUnits FROM %(table_name)s "
               "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL "
               "ORDER BY pm2_5 ASC LIMIT 1;",
        'max': "SELECT pm2_5, usUnits FROM %(table_name)s "
               "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL "
               "ORDER BY pm2_5 DESC LIMIT 1;",
        'sum': "SELECT SUM(pm2_5), usUnits FROM %(table_name)s "
               "WHERE dateTime > %(start)s AND dateTime <= %(stop)s AND pm2_5 IS NOT NULL)",
    }

    @staticmethod
    def compute_pm2_5_aqi(pm2_5):
        #             U.S. EPA PM2.5 AQI
        #
        #  AQI Category  AQI Value  24-hr PM2.5		AQIEEA	PM2.5
        # Good             0 -  50    0.0 -  12.0	0..1	< 10
        # Moderate        51 - 100   12.1 -  35.4	1..2	< 20
        # USG            101 - 150   35.5 -  55.4	2..3	< 25
        # Unhealthy      151 - 200   55.5 - 150.4	3..4	< 50
        # Very Unhealthy 201 - 300  150.5 - 250.4	4..5	< 75
        # Hazardous      301 - 400  250.5 - 350.4	>5	> 75
        # Hazardous      401 - 500  350.5 - 500.4

        if pm2_5 is None:
            return None

        # The EPA standard for AQI says to truncate PM2.5 to one decimal place.
        # See https://www3.epa.gov/airnow/aqi-technical-assistance-document-sept2018.pdf
        x = math.trunc(pm2_5 * 10) / 10

        if x <= 12.0: # Good
            return round(x / 12.0 * 50)
        elif x <= 35.4: # Moderate
            return round((x - 12.1) / 23.3 * 49.0 + 51.0)
        elif x <= 55.4: # Unhealthy for senstive
            return round((x - 35.5) / 19.9 * 49.0 + 101.0)
        elif x <= 150.4: # Unhealthy
            return round((x - 55.5) / 94.9 * 49.0 + 151.0)
        elif x <= 250.4: # Very Unhealthy
            return round((x - 150.5) / 99.9 * 99.0 + 201.0)
        elif x <= 350.4: # Hazardous
            return round((x - 250.5) / 99.9 * 99.0 + 301.0)
        else: # Hazardous
            return round((x - 350.5) / 149.9 * 99.0 + 401.0)

    @staticmethod
    def compute_pm2_5_aqi_color(pm2_5_aqi):
        if pm2_5_aqi is None:
            return None

        if pm2_5_aqi <= 50:
            return 128 << 8                 # Green
        elif pm2_5_aqi <= 100:
            return (255 << 16) + (255 << 8) # Yellow
        elif pm2_5_aqi <=  150:
            return (255 << 16) + (140 << 8) # Orange
        elif pm2_5_aqi <= 200:
            return 255 << 16                # Red
        elif pm2_5_aqi <= 300:
            return (128 << 16) + 128        # Purple
        else:
            return 128 << 16                # Maroon

    @staticmethod
    def compute_pm_2p5_us_epa_correction(pm_2p5: float, hum: float, temp: float) -> float:
        # 2021 EPA Correction
        # Low Concentration PAcf_1 ≤ 343 μg m-3  : PM2.5 = 0.52 x PAcf_1 - 0.086 x RH + 5.75
        # High Concentration PAcf_1 > 343 μg m-3 : PM2.5 = 0.46 x PAcf_1 + 3.93 x 10**-4 x PAcf_1**2 + 2.97
        #
        if pm_2p5 < 343.0:
            val = 0.52 * pm_2p5 - 0.086 * hum + 5.75
        else:
            val = 0.46 * pm_2p5 + 3.93 * 10**-4 * pm_2p5 ** 2 + 2.97

        return val if val >= 0.0 else 0.0

    @staticmethod
    def get_scalar(obs_type, record, db_manager=None):
        log.debug('get_scalar(%s)' % obs_type)
        if obs_type not in [ 'pm2_5_aqi', 'pm2_5_aqi_color' ]:
            raise weewx.UnknownType(obs_type)
        log.debug('get_scalar(%s)' % obs_type)
        if record is None:
            log.debug('get_scalar called where record is None.')
            raise weewx.CannotCalculate(obs_type)
        if 'pm2_5' not in record:
            # Should not see this as pm2_5 is part of the extended schema that is required for this plugin.
            # Returning CannotCalculate causes exception in ImageGenerator, return UnknownType instead.
            # ERROR weewx.reportengine: Caught unrecoverable exception in generator 'weewx.imagegenerator.ImageGenerator'
            log.info('get_scalar called where record does not contain pm2_5.  This is unexpected.')
            raise weewx.UnknownType(obs_type)
        if record['pm2_5'] is None:
            # Returning CannotCalculate causes exception in ImageGenerator, return UnknownType instead.
            # ERROR weewx.reportengine: Caught unrecoverable exception in generator 'weewx.imagegenerator.ImageGenerator'
            # Any archive catchup records will have None for pm2_5.
            log.debug('get_scalar called where record[pm2_5] is None: %s.  Probably a catchup record.' %
                timestamp_to_string(record['dateTime']))
            raise weewx.UnknownType(obs_type)
        try:
            pm2_5 = record['pm2_5']
            if obs_type == 'pm2_5_aqi':
                value = AQI.compute_pm2_5_aqi(pm2_5)
            if obs_type == 'pm2_5_aqi_color':
                value = AQI.compute_pm2_5_aqi_color(AQI.compute_pm2_5_aqi(pm2_5))
            t, g = weewx.units.getStandardUnitType(record['usUnits'], obs_type)
            # Form the ValueTuple and return it:
            return weewx.units.ValueTuple(value, t, g)
        except KeyError:
            # Don't have everything we need. Raise an exception.
            raise weewx.CannotCalculate(obs_type)

    @staticmethod
    def get_series(obs_type, timespan, db_manager, aggregate_type=None, aggregate_interval=None):
        """Get a series, possibly with aggregation.
        """

        if obs_type not in [ 'pm2_5_aqi', 'pm2_5_aqi_color' ]:
            raise weewx.UnknownType(obs_type)

        log.debug('get_series(%s, %s, %s, aggregate:%s, aggregate_interval:%s)' % (
            obs_type, timestamp_to_string(timespan.start), timestamp_to_string(
            timespan.stop), aggregate_type, aggregate_interval))

        #  Prepare the lists that will hold the final results.
        start_vec = list()
        stop_vec = list()
        data_vec = list()

        # Is aggregation requested?
        if aggregate_type:
            # Yes. Just use the regular series function.
            return weewx.xtypes.ArchiveTable.get_series(obs_type, timespan, db_manager, aggregate_type,
                                           aggregate_interval)
        else:
            # No aggregation.
            sql_str = 'SELECT dateTime, usUnits, `interval`, pm2_5 FROM %s ' \
                      'WHERE dateTime >= ? AND dateTime <= ? AND pm2_5 IS NOT NULL' \
                      % db_manager.table_name
            std_unit_system = None

            for record in db_manager.genSql(sql_str, timespan):
                ts, unit_system, interval, pm2_5 = record
                if std_unit_system:
                    if std_unit_system != unit_system:
                        raise weewx.UnsupportedFeature(
                            "Unit type cannot change within a time interval.")
                else:
                    std_unit_system = unit_system

                if obs_type == 'pm2_5_aqi':
                    value = AQI.compute_pm2_5_aqi(pm2_5)
                if obs_type == 'pm2_5_aqi_color':
                    value = AQI.compute_pm2_5_aqi_color(AQI.compute_pm2_5_aqi(pm2_5))
                log.debug('get_series(%s): %s - %s - %s' % (obs_type,
                    timestamp_to_string(ts - interval * 60),
                    timestamp_to_string(ts), value))
                start_vec.append(ts - interval * 60)
                stop_vec.append(ts)
                data_vec.append(value)

            unit, unit_group = weewx.units.getStandardUnitType(std_unit_system, obs_type,
                                                               aggregate_type)

        return (ValueTuple(start_vec, 'unix_epoch', 'group_time'),
                ValueTuple(stop_vec, 'unix_epoch', 'group_time'),
                ValueTuple(data_vec, unit, unit_group))

    @staticmethod
    def get_aggregate(obs_type, timespan, aggregate_type, db_manager, **option_dict):
        """Returns an aggregation of pm2_5_aqi over a timespan by using the main archive
        table.

        obs_type

        timespan: An instance of weeutil.Timespan with the time period over which aggregation is to
        be done.

        aggregate_type: The type of aggregation to be done. For this function, must be 'avg',
        'sum', 'count', 'first', 'last', 'min', or 'max'. Anything else will cause
        weewx.UnknownAggregation to be raised.

        db_manager: An instance of weewx.manager.Manager or subclass.

        option_dict: Not used in this version.

        returns: A ValueTuple containing the result.
        """
        if obs_type not in [ 'pm2_5_aqi', 'pm2_5_aqi_color' ]:
            raise weewx.UnknownType(obs_type)

        log.debug('get_aggregate(%s, %s, %s, aggregate:%s)' % (
            obs_type, timestamp_to_string(timespan.start),
            timestamp_to_string(timespan.stop), aggregate_type))

        aggregate_type = aggregate_type.lower()

        # Raise exception if we don't know about this type of aggregation
        if aggregate_type not in list(AQI.agg_sql_dict.keys()):
            raise weewx.UnknownAggregation(aggregate_type)

        # Form the interpolation dictionary
        interpolation_dict = {
            'start': timespan.start,
            'stop': timespan.stop,
            'table_name': db_manager.table_name
        }

        select_stmt = AQI.agg_sql_dict[aggregate_type] % interpolation_dict
        row = db_manager.getSql(select_stmt)
        if row:
            value, std_unit_system = row
        else:
            value = None
            std_unit_system = None

        if value is not None:
            if obs_type == 'pm2_5_aqi':
                value = AQI.compute_pm2_5_aqi(value)
            if obs_type == 'pm2_5_aqi_color':
                value = AQI.compute_pm2_5_aqi_color(AQI.compute_pm2_5_aqi(value))
        t, g = weewx.units.getStandardUnitType(std_unit_system, obs_type, aggregate_type)
        # Form the ValueTuple and return it:
        log.debug('get_aggregate(%s, %s, %s, aggregate:%s, select_stmt: %s, returning %s)' % (
            obs_type, timestamp_to_string(timespan.start), timestamp_to_string(timespan.stop),
            aggregate_type, select_stmt, value))
        return weewx.units.ValueTuple(value, t, g)

if __name__ == "__main__":
    usage = """%prog [options] [--help] [--debug]"""

    import weeutil.logger

    def main():
        import optparse
        parser = optparse.OptionParser(usage=usage)
        parser.add_option('--config', dest='cfgfn', type=str, metavar="FILE",
                          help="Use configuration file FILE. Default is /etc/weewx/weewx.conf or /home/weewx/weewx.conf")
        parser.add_option('--test-extension', dest='te', action='store_true',
                          help='test the data collector')
        parser.add_option('--hostname', dest='hostname', action='store',
                          help='hostname to use with --test-collector')
        parser.add_option('--port', dest='port', action='store',
                          type=int, default=80,
                          help="port to use with --test-collector. Default is '80'")
        (options, args) = parser.parse_args()

        weeutil.logger.setup('airlink', {})

        if options.te:
            if not options.hostname:
                parser.error('--test-collector requires --hostname argument')
            test_extension(options.hostname, options.port)

    def test_extension(hostname, port):
        sources = [Source({'Sensor1': { 'enable': True, 'hostname': hostname, 'port': port, 'timeout': 2}}, 'Sensor1')]
        cfg = Configuration(
            lock             = threading.Lock(),
            concentrations   = None,
            archive_interval = 300,
            archive_delay    = 15,
            poll_interval    = 5,
            sources          = sources)
        while True:
            with cfg.lock:
                cfg.concentrations = get_concentrations(cfg)
            print('%s:%d concentrations: %s' % (cfg.sources[0].hostname, cfg.sources[0].port, cfg.concentrations))
            packet = {}
            AirLink.fill_in_packet(cfg, packet)
            print('Fields to be inserted into packet: %s' % packet)
            time.sleep(cfg.poll_interval)

    main()
