# Installer file for WeatherLink Live, Airlink and Health (WeatherLinkLiveUDP) driver for WeeWX
# Copyright 2022 Werner Krenn
# Modified 2021/2022 Werner Krenn (leaf/soil/Extra2..4,ISS2,Wind,Rain,Batlevel,Signal)
# Distributed under the terms of the GNU Public License (GPLv3)

# python imports
import configobj
from distutils.version import StrictVersion
from setup import ExtensionInstaller

# import StringIO, use six.moves due to python2/python3 differences
from six.moves import StringIO

# WeeWX imports
import weewx

REQUIRED_VERSION = "3.7.0"
LIVEUDP_VERSION = "0.5.3"
# define our config as a multiline string so we can preserve comments
liveudp_config = """
[Station]
    station_type = WeatherLinkLiveUDP

[StdCalibrate]
    [[Corrections]]
        #rxCheckPercent = signalQuality1(data_binding=davishealthapi_binding)
        co2_Temp = temp if temp is not None else None
        co2_Hum = hum if hum is not None else None

[DataBindings]
    [[wx_binding]]
        #schema = schemas.wview_extended2my.schema
    
    #[[davishealthapi_binding]]
    #    database = davishealthapi_sqlite
    #    table_name = archive
    #    manager = weewx.manager.DaySummaryManager
    #    schema = user.davishealthapi.schema

[Databases]
    [[archive_sqlite]]
        #database_name = weewx_wlludp.sdb
    #[[davishealthapi_sqlite]]
    #   database_type = SQLite
    #   database_name = davishealthapi.sdb

[StdWXCalculate]
    [[Calculations]]
        GTS = software, archive
        GTSdate = software, archive
        utcoffsetLMT = software, archive
        dayET = prefer_hardware, archive
        ET24 = prefer_hardware, archive
        yearGDD = software, archive
        seasonGDD = software, archive

[Engine]
    [[Services]]
        #data_services = user.airlink.AirLink, user.davishealthapi.DavisHealthAPI
        #data_services = user.airlink.AirLink,
        xtype_services = weewx.wxxtypes.StdWXXTypes, weewx.wxxtypes.StdPressureCooker, weewx.wxxtypes.StdRainRater, weewx.wxxtypes.StdDelta, user.GTS.GTSService
        process_services = weewx.engine.StdConvert, weewx.engine.StdCalibrate, weewx.engine.StdQC, weewx.wxservices.StdWXCalculate, user.sunrainduration.SunshineDuration, user.radiationhours.RadiationHours

[StdReport]
    [[AirLinkReport]]
        HTML_ROOT = /var/www/html/weewx/airlink
        enable = true
        skin = airlink
    
    [[DavisHealth]]
        HTML_ROOT = /var/www/html/weewx/health
        enable = false
        skin = health

[WeatherLinkLiveUDP]
  driver = user.weatherlinkliveudp
  wll_ip = 1.2.3.4
  poll_interval =  30
  txid_iss = 1
  extra_id = 0                  #0 = not available - Station ID in Live (1..8)
  extra_id2 = 0
  extra_id3 = 0
  extra_id4 = 0
  leaf = 0
  soil = 0
  wind = 0
  txid_rain = 0
  txid_iss2 = 0
  #did = 001D0A000000           #MAC-Adress of the Live - is needed, if more then one DAVIS stations reports at port 22222
  log = 0                       #internal log-level: 1=UDP check time, 2=only archive-packets, 3=all packets, 4=iss, 5=extra_data1..4, 6=Wind,Rain, 7=ISS2, 8=iss_udp, 9=all reception

[AirLink]
    [[Sensor1]]
        enable = True
        hostname = 192.168.0.145
        port = 80
        timeout = 2
    [[Sensor2]]
        enable = False
        hostname = airlink2
        port = 80
        timeout = 2

[davishealthapi]
    data_binding = davishealthapi_binding
    api_secret = ????????????????????????????????
    api_key = ????????????????????????????????
    station_id = ??????
    max_age = None                  # default is 2592000
    #max_count = 13                 # search for 13 sensors (8 Live, 1 indoor, 1 baro, 1 AirLink, 1 Health Live, 1 AirLink Health Record )
                                    # or max_count = 7 search only for 7 sensors (was the default from the origin davishealthapi)
                                    # or max_count = 0  or not set =  automatical count the available sensors
    packet_log = 0

    #Is for sensor_tx1 = 0 set, the program seach for the first found sensor and use it, 
    #so it's mostly better to set here the tx_ID of your VUE (or VantagePro)
    #If at the other (sensor_tx2, sensor_tx3 ...) 0 set, the program don't search and use such a sensor.  
    #If a DAVIS AirLink is also attached at the DAVIS Weatherlink Live, it's automatical used. 
    sensor_tx1 = 1                  #ISS
    sensor_tx2 = 0
    sensor_tx3 = 0
    sensor_tx4 = 0
    sensor_tx5 = 0
    sensor_tx6 = 0
    sensor_tx7 = 0
    sensor_tx8 = 0

[RadiationDays]
    min_sunshine = 120
    sunshine_coeff = 0.8
    sunshine_min = 18
    sunshine_loop = 1
    rainDur_loop = 0
    sunshine_log = 0
    rainDur_log = 0

    rain2 = 0
    sunshine2 = 0    
    sunshine2_loop = 1
    rainDur2_loop = 0
    sunshine2_log = 0
    rainDur2_log = 0

"""

# construct our config dict
liveupd_dict = configobj.ConfigObj(StringIO(liveudp_config))


def loader():
    return LiveUDPInstaller()


class LiveUDPInstaller(ExtensionInstaller):
    def __init__(self):
        if StrictVersion(weewx.__version__) < StrictVersion(REQUIRED_VERSION):
            msg = "%s requires WeeWX %s or greater, found %s" % (''.join(('WeatherLinkLiveUDP driver ', LIVEUPD_VERSION)),
                                                                 LIVEUPD_VERSION,
                                                                 weewx.__version__)
            raise weewx.UnsupportedFeature(msg)

        super(LiveUDPInstaller, self).__init__(
            version='0.5.1',
            name='WeatherLinkLiveUDP',
            description='Periodically poll weather data from a WeatherLink Live device',
            author="Bastiaan Meelberg, Werner Krenn",
#           files=[('bin/user', ['bin/user/weatherlinkliveudp.py'])],
            files=[
                ('bin/user', [
                    'bin/user/weatherlinkliveudp.py',
                    'bin/user/airlink.py',
                    'bin/user/davishealthapi.py',
                    'bin/user/extensions.py',
                    'bin/user/historygenerator.py',
                    'bin/user/GTS.py',
                    'bin/user/dayboundarystats.py',
                    'bin/user/radiationhours.py',
                    'bin/user/sunrainduration.py',
                ]),
                ('bin/schemas', [
                    'bin/schemas/wview_extended2my.py',
                ]),
                ('import', [
                    'import/csv-wswin0.conf',
                    'import/csv-wswin.conf',
                ]),
                ('skins/nws', [
                    'skins/Seasons/index.html.tmpl',
                    'skins/Seasons/skin.conf',
                    'skins/Seasons/seasons.css',
                    'skins/Seasons/seasons.js',
                    'skins/Seasons/current.inc',
                    'skins/Seasons/hilo.inc',
                    'skins/Seasons/sensors.inc',
                    'skins/Seasons/statistics.inc',
                    'skins/Seasons/airlink.inc',
                    'skins/Seasons/health.inc',
                    'skins/Seasons/airlink.inc',
                    'skins/Seasons/belchertown.inc',
                    'skins/Seasons/neowx.inc',
                    'skins/airlink/index.html.tmpl',
                    'skins/airlink/skin.conf',
                    'skins/airlink/titlebar.inc',
                    'skins/health/index.html.tmpl',
                    'skins/health/skin.conf',
                    'skins/health/health.css',
                    'skins/health/health.js',
                    'skins/health/favicon.ico',
                    'skins/health/identifier.inc',
                    'skins/health/titlebar.inc',
                    'skins/health/sensorsair.inc',
                    'skins/health/sensor.inc',
                ]),
            ],
            config=liveudp_dict
         )

